"use client";
import React, { useState } from "react";
import "./ContactSec.css";
import Image from "next/image";

const ContactSec = () => {
  const budgetOptions = [
    { id: "1", label: "Less than $5K" },
    { id: "2", label: "$5K - $10K" },
    { id: "3", label: "$10K - $20K" },
    { id: "4", label: "$20K - $50K" },
    { id: "5", label: "More than $50K" },
  ];

  const [formData, setFormData] = useState({
    fullName: "",
    email: "",
    phone: "",
    budget: "",
    message: "",
    pageLink: "",
  });

  const [selected, setSelected] = useState("");
  const [loading, setLoading] = useState(false);
  const [status, setStatus] = useState("");

  const features = [
    "Expect a response from us within 60 Minutes",
    "We're happy to sign an NDA upon request.",
    "Get access to a team of dedicated product specialists.",
  ];

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData((prev) => ({ ...prev, [name]: value }));
  };

  const handleBudgetSelect = (id, label) => {
    setSelected(id);
    setFormData((prev) => ({ ...prev, budget: label }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setStatus("");

    const phoneRegex = /^[+]?[0-9]{10,15}$/;
    if (formData.phone && !phoneRegex.test(formData.phone)) {
      setStatus("Please enter a valid phone number.");
      setLoading(false);
      return;
    }

    const currentPageLink =
      typeof window !== "undefined" ? window.location.href : "";

    const updatedFormData = { ...formData, pageLink: currentPageLink };

    try {
      const res = await fetch("/api/sendEmail", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(updatedFormData),
      });

      const data = await res.json();

      if (res.ok) {
        setStatus("Your message has been sent successfully!");
        setFormData({
          fullName: "",
          email: "",
          phone: "",
          budget: "",
          message: "",
          pageLink: "",
        });
        setSelected("");
      } else {
        setStatus(data.message || "Something went wrong.");
      }

      setTimeout(() => setStatus(""), 3000);
    } catch (error) {
      setStatus("Failed to send email. Please try again later.");
      setTimeout(() => setStatus(""), 3000);
    } finally {
      setLoading(false);
    }
  };

  return (
    <section className="contact-section">
      <div className="container">
        <div className="contact-wrapper">
          <div className="row align-items-center">
            {/* Left Section - Hero */}
            <div className="col-12 col-lg-6">
              <div className="contact-hero">
                <div className="contact-heading">
                  <h1 className="heading-orange">Tell Us</h1>
                  <h1 className="heading-black">Your Amazing</h1>
                  <h1 className="heading-orange">Project Here</h1>
                </div>

                <div className="features-list">
                  {features.map((feature, index) => (
                    <div key={index} className="feature-item">
                      <p>{feature}</p>
                    </div>
                  ))}
                </div>

                {/* Project Showcase */}
                <div className="project-showcase">
                  <div className="blur-effect">
                    <svg
                      width="790"
                      height="210"
                      viewBox="0 0 790 210"
                      fill="none"
                      xmlns="http://www.w3.org/2000/svg"
                    >
                      <g filter="url(#filter0_f_0_73)">
                        <ellipse
                          cx="395"
                          cy="105"
                          rx="327"
                          ry="37"
                          fill="#FFA64D"
                        />
                      </g>
                      <defs>
                        <filter
                          id="filter0_f_0_73"
                          x="0.900002"
                          y="0.900002"
                          width="788.2"
                          height="208.2"
                          filterUnits="userSpaceOnUse"
                          colorInterpolationFilters="sRGB"
                        >
                          <feFlood
                            floodOpacity="0"
                            result="BackgroundImageFix"
                          />
                          <feBlend
                            mode="normal"
                            in="SourceGraphic"
                            in2="BackgroundImageFix"
                            result="shape"
                          />
                          <feGaussianBlur
                            stdDeviation="33.55"
                            result="effect1_foregroundBlur_0_73"
                          />
                        </filter>
                      </defs>
                    </svg>
                  </div>
                  <div className="books-container">
                    <Image
                      src="/contactBook.webp"
                      alt="Main book presentation"
                      width={789}
                      height={298}
                      className="book-img book-main"
                    />
                  </div>
                </div>
              </div>
            </div>

            {/* Right Section - Form */}
            <div className="col-12 col-lg-6">
              <div className="contact-form-wrappers">
                <form onSubmit={handleSubmit} className="contact-form">
                  <input
                    type="hidden"
                    name="pageLink"
                    value={formData.pageLink}
                  />

                  <div className="form-group">
                    <label htmlFor="fullName">Full Name</label>
                    <input
                      type="text"
                      id="fullName"
                      name="fullName"
                      value={formData.fullName}
                      onChange={handleChange}
                      placeholder="Jhon Deo"
                      required
                      className="form-control"
                    />
                  </div>

                  <div className="row g-3">
                    <div className="col-12 col-md-6">
                      <div className="form-group">
                        <label htmlFor="email">Your Email</label>
                        <input
                          type="email"
                          id="email"
                          name="email"
                          value={formData.email}
                          onChange={handleChange}
                          placeholder="yourmail@gmail.com"
                          required
                          className="form-control"
                        />
                      </div>
                    </div>
                    <div className="col-12 col-md-6">
                      <div className="form-group">
                        <label htmlFor="phone">Phone no</label>
                        <input
                          type="tel"
                          id="phone"
                          name="phone"
                          value={formData.phone}
                          onChange={handleChange}
                          placeholder="xxx xxx xxxx"
                          pattern="^[+]?[0-9]{10,15}$"
                          title="Enter a valid phone number (10–15 digits)"
                          required
                          className="form-control"
                        />
                      </div>
                    </div>
                  </div>

                  <div className="form-group">
                    <label>Project Budget</label>
                    <div className="budget-options">
                      {budgetOptions.map((budget) => (
                        <button
                          key={budget.id}
                          type="button"
                          onClick={() =>
                            handleBudgetSelect(budget.id, budget.label)
                          }
                          className={`budget-btn ${
                            selected === budget.id ? "active" : ""
                          }`}
                        >
                          {budget.label}
                        </button>
                      ))}
                    </div>
                  </div>

                  <div className="form-group">
                    <label htmlFor="message">Project Details</label>
                    <textarea
                      id="message"
                      name="message"
                      value={formData.message}
                      onChange={handleChange}
                      placeholder="Write Message....."
                      required
                      rows={6}
                      className="form-control"
                    />
                  </div>

                  <button
                    type="submit"
                    disabled={loading}
                    className="submit-btn"
                  >
                    <span>{loading ? "Sending..." : "Let's Connect"}</span>
                    {!loading && (
                      <svg
                        width="25"
                        height="25"
                        viewBox="0 0 25 25"
                        fill="none"
                        xmlns="http://www.w3.org/2000/svg"
                      >
                        <path
                          d="M16.6826 9.81125L8.41994 18.074L7.0625 16.7165L15.3242 8.45381H8.04266V6.53381H18.6026V17.0938H16.6826V9.81125Z"
                          fill="white"
                        />
                      </svg>
                    )}
                  </button>

                  {loading && (
                    <div className="loader-container">
                      <div className="loader"></div>
                    </div>
                  )}

                  {status && (
                    <div className="status-message-container">
                      <p
                        className={`status-message ${!status ? "hidden" : ""}`}
                      >
                        {status}
                      </p>
                    </div>
                  )}
                </form>
              </div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
};

export default ContactSec;
