"use client";
import React, { useState } from "react";
import "./ContactForm.css";

export default function ContactForm() {
  const [formData, setFormData] = useState({
    name: "",
    email: "",
    phone: "",
    message: "",
  });

  const [loading, setLoading] = useState(false);
  const [status, setStatus] = useState("");

  const handleChange = (e) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value,
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setStatus("");

    const phoneRegex = /^[+]?[0-9]{10,15}$/;
    if (formData.phone && !phoneRegex.test(formData.phone)) {
      setStatus("Please enter a valid phone number.");
      setLoading(false);
      return;
    }

    const currentPageLink =
      typeof window !== "undefined" ? window.location.href : "";

    const apiFormData = {
      fullName: formData.name,
      email: formData.email,
      phone: formData.phone,
      budget: "",
      message: formData.message,
      pageLink: currentPageLink,
    };

    try {
      const res = await fetch("/api/sendEmail", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(apiFormData),
      });

      const data = await res.json();

      if (res.ok) {
        setStatus("Your message has been sent successfully!");
        setFormData({
          name: "",
          email: "",
          phone: "",
          message: "",
        });
      } else {
        setStatus(data.message || "Something went wrong.");
      }

      setTimeout(() => setStatus(""), 3000);
    } catch (error) {
      setStatus("Failed to send email. Please try again later.");
      setTimeout(() => setStatus(""), 3000);
    } finally {
      setLoading(false);
    }
  };

  return (
    <>
      <section className="banner-form">
        <div className="container-fluid d-flex justify-content-center align-items-center py-4">
          <main className="contact-form-container p-4 p-md-5 w-100 d-flex flex-column justify-content-center align-items-center">
            <div className="w-100" style={{ maxWidth: "1544px" }}>
              <form onSubmit={handleSubmit}>
                <div className="row">
                  {/* Header Section */}
                  <div className="col-12 mb-4">
                    <header className="text-center">
                      <h1 className="title-main mb-">
                        Success is Waiting! Avail Upto 50% Off Today
                      </h1>
                      <h2 className="subtitle mb-0">On Our All Services</h2>
                    </header>
                  </div>

                  {/* Input Fields Row */}
                  <div className="col-12 mb-3">
                    <div className="row g-3 g-md-4">
                      <div className="col-12 col-md-4">
                        <input
                          type="text"
                          name="name"
                          placeholder="Name"
                          className="form-control form-control-custom"
                          value={formData.name}
                          onChange={handleChange}
                          required
                        />
                      </div>
                      <div className="col-12 col-md-4">
                        <input
                          type="email"
                          name="email"
                          placeholder="Email"
                          className="form-control form-control-custom"
                          value={formData.email}
                          onChange={handleChange}
                          required
                        />
                      </div>
                      <div className="col-12 col-md-4">
                        <input
                          type="tel"
                          name="phone"
                          placeholder="Phone Number"
                          className="form-control form-control-custom"
                          value={formData.phone}
                          onChange={handleChange}
                          pattern="^[+]?[0-9]{10,15}$"
                          title="Enter a valid phone number (10–15 digits)"
                          required
                        />
                      </div>
                    </div>
                  </div>

                  {/* Textarea Row */}
                  <div className="col-12 mb-4">
                    <textarea
                      name="message"
                      placeholder="To help us understand better, enter a brief  message about your project"
                      className="form-control form-control-textarea"
                      value={formData.message}
                      onChange={handleChange}
                      required
                    />
                  </div>

                  {/* Submit Button Row */}
                  <div className="col-12">
                    <button
                      type="submit"
                      className="btn btn-submit w-100"
                      disabled={loading}
                    >
                      {loading ? "Sending..." : "Submit"}
                    </button>
                  </div>

                  {/* Loader */}
                  {loading && (
                    <div className="col-12 mt-3">
                      <div className="loader-container">
                        <div className="loader"></div>
                      </div>
                    </div>
                  )}

                  {/* Status Message */}
                  {status && (
                    <div className="col-12 mt-3">
                      <p
                        className={`status-message ${!status ? "hidden" : ""}`}
                      >
                        {status}
                      </p>
                    </div>
                  )}
                </div>
              </form>
            </div>
          </main>
        </div>
      </section>
    </>
  );
}
