"use client";
import axiosInstance from "@/utilis/axiosInstance";
import Link from "next/link";
import { useEffect, useState } from "react";
import "./BlogSection.css";
import Image from "next/image";

const BlogSection = () => {
  const [blogs, setBlogs] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    const fetchBlogs = async () => {
      try {
        setLoading(true);
        const response = await axiosInstance.get("/api/blogs");

        setBlogs(response.data.items);
        setError(null);
      } catch (err) {
        console.error("Error fetching blogs:", err);
        setError("Failed to load blogs. Please try again later.");
      } finally {
        setLoading(false);
      }
    };

    fetchBlogs();
  }, []);

  if (loading) {
    return (
      <div className="blog-section-wrapper">
        <div className="container text-center py-5">
          <div className="spinner-border text-primary" role="status">
            <span className="visually-hidden">Loading...</span>
          </div>
          <p className="mt-3">Loading blogs...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="blog-section-wrapper">
        <div className="container text-center py-5">
          <div className="alert alert-danger" role="alert">
            {error}
          </div>
        </div>
      </div>
    );
  }

  return (
    <>
      <div className="blog-section-wrapper">
        <div className="container">
          {/* Recent Blog Posts Section */}

          <section className="recent-posts-section mb-5">
            <header className="section-header mb-4">
              <h2 className="section-title">
                Recent{" "}
                <span className="section-title-highlight">Blog Posts</span>
              </h2>
            </header>

            <div className="row g-4">
              {/* Featured Blog Card */}

              <div className="col-12 col-lg-6">
                {blogs?.[0] && (
                  <article className="blog-card featured-card">
                    <Link href={`/blogs/${blogs[0].slug}`}>
                      <Image
                        src={blogs[0]?.blogImage || "/images/logo.webp"}
                        alt={blogs[0].title}
                        width={600}
                        height={400}
                        className="blog-card-img"
                      />
                      <div className="blog-card-content">
                        <time className="blog-date">
                          {new Date(blogs[0]?.createdAt).toLocaleDateString(
                            "en-US",
                            {
                              weekday: "long",
                              day: "numeric",
                              month: "short",
                              year: "numeric",
                            }
                          )}
                        </time>
                        <div className="d-flex align-items-center gap-3 mb-3">
                          <h3 className="blog-title featured-title text-uppercase">
                            {blogs[0].title}
                          </h3>
                        </div>
                        <p className="blog-description featured-description">
                          {blogs[0]?.content
                            ? blogs[0]?.content
                                .replace(/<[^>]+>/g, "")
                                .slice(0, 120) + "..."
                            : ""}
                        </p>
                        <button className="btn-read-more">Read More</button>
                      </div>
                    </Link>
                  </article>
                )}
              </div>

              <div className="col-12 col-lg-6">
                {blogs?.slice(1, 3).map((post) => (
                  <div key={post._id} className="compact-cards-wrapper">
                    <article className="blog-card compact-card">
                      <Link href={`/blogs/${post.slug}`}>
                        <div className="row g-3 mb-5">
                          <div className="col-12 col-md-5">
                            <Image
                              src={post?.blogImage || "/images/logo.webp"}
                              alt={post.title}
                              width={300}
                              height={200}
                              className="blog-card-img compact-img"
                            />
                          </div>
                          <div className="col-12 col-md-7">
                            <div className="blog-card-content compact-content">
                              <time className="blog-date compact-date">
                                {new Date(post?.createdAt).toLocaleDateString(
                                  "en-US",
                                  {
                                    weekday: "long",
                                    day: "numeric",
                                    month: "short",
                                    year: "numeric",
                                  }
                                )}
                              </time>
                              <h3 className="blog-title compact-title">
                                {post.title}
                              </h3>
                              <p className="blog-description compact-description">
                                {post?.content
                                  ? post?.content
                                      .replace(/<[^>]+>/g, "")
                                      .slice(0, 40) + "..."
                                  : ""}
                              </p>
                              <button className="btn-read-more compact-btn">
                                Read More
                              </button>
                            </div>
                          </div>
                        </div>
                      </Link>
                    </article>
                  </div>
                ))}
              </div>
            </div>
          </section>

          {/* Gradient Divider */}
          {blogs.length > 3 && (
            <div className="gradient-divider my-5">
              <svg
                width="100%"
                height="4"
                viewBox="0 0 1599 4"
                fill="none"
                preserveAspectRatio="none"
              >
                <path
                  d="M0 2.42114H1598.5"
                  stroke="url(#paint0_linear)"
                  strokeWidth="3"
                />
                <defs>
                  <linearGradient
                    id="paint0_linear"
                    x1="0"
                    y1="2.92114"
                    x2="1598.5"
                    y2="2.92114"
                    gradientUnits="userSpaceOnUse"
                  >
                    <stop stopColor="#F96805" stopOpacity="0" />
                    <stop offset="0.504808" stopColor="#F96805" />
                    <stop offset="1" stopColor="#F96805" stopOpacity="0" />
                  </linearGradient>
                </defs>
              </svg>
            </div>
          )}

          {/* All Blog Posts Section */}

          <section className="all-posts-section">
            <header className="section-header mb-4">
              <h2 className="section-title">
                All <span className="section-title-highlight">Blog Posts</span>
              </h2>
            </header>

            <div className="row g-4 mb-5">
              {blogs?.map((post) => {
                const imageSrc =
                  post?.blogImage?.startsWith("http") && post.blogImage
                    ? post.blogImage
                    : "/images/logo.webp";
                return (
                  <div key={post._id} className="col-12 col-md-6 col-lg-4">
                    <article className="blog-card all-posts-card">
                      <Link href={`/blogs/${post.slug}`}>
                        <Image
                          src={imageSrc}
                          alt={post.title}
                          width={300}
                          height={200}
                          className="blog-card-img all-posts-img"
                        />
                        <div className="blog-card-content">
                          <time className="blog-date">
                            {new Date(
                              post.createdAt || post.date
                            ).toLocaleDateString("en-US", {
                              weekday: "long",
                              year: "numeric",
                              month: "short",
                              day: "numeric",
                            })}
                          </time>
                          <div className="d-flex align-items-start gap-3 mb-3">
                            <h3 className="blog-title all-posts-title flex-grow-1">
                              {post.title}
                            </h3>
                          </div>
                          <p className="blog-description all-posts-description">
                            {post?.content
                              ? post.content
                                  .replace(/<[^>]+>/g, "")
                                  .slice(0, 120) + "..."
                              : post?.title}
                          </p>
                        </div>
                      </Link>
                    </article>
                  </div>
                );
              })}
            </div>
          </section>
        </div>
      </div>
    </>
  );
};

export default BlogSection;
